clc, clear, close all
%% === Add paths ==========================================================
addpath('./Dlo');
addpath('./Domains');
addpath('./Macro Element');
addpath('./Plot');
addpath('./Pore Pressure');
addpath('./Self Weight');

%% === Tolerances =========================================================
ColTol = 1-(1e-6);
Cutoff = 0.002; Ng = 50; % Plot: Member Cutoff & Number of plot groups
eps = 1e-5;
Nmax = 20;

%% == Examples ============================================================

% --- OPTION 1: Bearing Capacity ------------------------------------------
RefineLvl = 1;
MElvl = 1;
[NODE, ELEM, SUPP, LOAD, HYDH, EMAT] = ...
 BearingCapacityDomain(RefineLvl);
SFl = 0.1;
SFu = 5.;
MatProp = [20. 0*pi/180 1e-6]; % Soil 1
gammaw = 10;

% --- OPTION 2: Layeded Slope ---------------------------------------------
% RefineLvl = 1;
% MElvl = 1;
% [NODE, ELEM, SUPP, LOAD, HYDH, EMAT] = LayeredSlopeDomain(RefineLvl);
% SFl = 0.1;
% SFu = 5.;
% MatProp = [0.0 38*pi/180 19.5; % Soil 1
%            5.3 23*pi/180 19.5; % Soil 2
%            7.2 20*pi/180 19.5];% Soil 3
% gammaw = 10;

% --- OPTION 3: Earth dam -------------------------------------------------
% RefineLvl = 2;
% MElvl = 2;
% [NODE, ELEM, SUPP, LOAD, HYDH, EMAT] = DuncanWrightDomain(RefineLvl);
% SFl = 0.1;
% SFu = 5.;
% MatProp = [0.0 38*pi/180 140 1.67*1e-5; % Soil 1
%            0.0 20*pi/180 120 1.67*1e-7];% Soil 2
% gammaw = 64.23;

% --- OPTION 4: Multi material ground water -------------------------------
% RefineLvl = 1;
% MElvl = 1; 
% [NODE, ELEM, SUPP, LOAD, HYDH, EMAT] = ...
%     MultiMaterialGroundWaterDomain(RefineLvl);
% SFl = 0.1;
% SFu = 5.;
% MatProp = [0 20*pi/180 18.82 1.67*1e-5; % Soil 1
%            5.3  20*pi/180 18.82 1.67*1e-5; % Soil 2
%            10.3 20*pi/180 18.82 1.67*1e-5];% Soil 3
% gammaw = 10;

%% === Domain Properties ==================================================
% --- Domain plot ---------------------------------------------------------
PlotMeshGeo(NODE, ELEM, SUPP, LOAD, HYDH, EMAT)

% --- Pore pressure ------------------------------------------------------- 
[uw] = PorePressure(NODE, ELEM, HYDH, EMAT, MatProp, gammaw);
[A,B] = MarchingElem(NODE, ELEM, uw, 0.0); 
PlotMeshPressure(NODE, ELEM, uw, A, B);

% --- Discontinuities ----------------------------------------------------- 
[BARS, NODEme, ELEMme, SUPPme, LOADme, HYDHme, uwme, BARMAT] = ...
    GenerateGSMEgeo(NODE, ELEM, SUPP, LOAD, HYDH, uw, MElvl, EMAT, MatProp);

% --- ME Domain plot ------------------------------------------------------
PlotMeshGeo(NODEme, ELEMme, SUPPme, LOADme, HYDHme, EMAT)

% --- Nodal weight --------------------------------------------------------

ELEMgamma = MatProp(EMAT, 3);
if isempty(HYDH)
    weightNODE = GetNodeWeight(NODEme,NODE,ELEM,gammaw,ELEMgamma);
else
    P = unique([ NODE(HYDH(:,1),1), HYDH(:,2)],'rows');
    weightNODE = GetNodeWeight(NODEme,NODE,ELEM,gammaw,ELEMgamma,[P; A; B]);
end
PlotMeshWeight(NODEme, ELEMme, weightNODE);

% --- Boundary ------------------------------------------------------------
[boundBARS,suppBOUND,loadBOUND,loadBOUNDvals] = ...
    GetGeoBoundary(BARS,NODEme,ELEMme,SUPPme,LOADme);

%% === Optimization =======================================================
% --- DLO matrices --------------------------------------------------------
[B,B0,N,I,fD,fL,fL0,g,Uw] = ...
    GetDLOmatrices(NODEme,BARS,BARMAT,boundBARS,suppBOUND,loadBOUND,...
                   loadBOUNDvals, weightNODE,uwme, 1); 
% --- Critical SF Search --------------------------------------------------------
Nn = length(NODE); Ne = length(ELEM); Nb = length(BARS); Np = size(N, 2);
fprintf('\n')
fprintf('Mesh: Elements %d, Nodes %d, Discont %d, ME level %d\n',Ne,Nn,Nb,MElvl)
tic
[S, fval, exitflag, SF] = ...
    DLOCriticalSFSearch(B,B0,N,I,fD,fL,g,Uw, SFl, SFu, Nmax, eps);
if (exitflag~=1), fprintf('--- WARNING: search did not converge.\n'); end
fprintf('Objective = %f\nSF = %f\nSF Search CPU time = %g s\n',fval,SF,toc);
fprintf('\n')
%% == Results =============================================================

d = S(1:2*Nb, 1);
p = S(2*Nb+1:2*Nb +Np, 1);
d = reshape(d, 2, numel(d)/2)';  
p = reshape(p, 2, numel(p)/2)'; 

% ---- Shear displacement -------------------------------------------------
s = d(:, 1);
PlotDLOResults(NODEme, BARS, s, Cutoff, Ng)
title('Shear Displacements')
PlotBoundary(ELEM, NODE)

% ---- Normal displacemet -------------------------------------------------
n = d(:, 2);
PlotDLOResults(NODEme, BARS, n, Cutoff, Ng)
title('Normal Displacements')
PlotBoundary(ELEM, NODE)

%% === Remove paths =======================================================
rmpath('./Dlo');
rmpath('./Domains');
rmpath('./Macro Element');
rmpath('./Plot');
rmpath('./Pore Pressure');
rmpath('./Self Weight');